/*
 *  JFP - "Just F***ing Patch" file patching system
 *
 *  Copyright (C) 2010 Hextator
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  <Description> Applies and creates patches in the JFP format
 *  with a minimal interface
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.security.MessageDigest;
import java.util.ArrayList;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.UIManager;

public class JFP {
	// General utility functions

	// XXX Refactor initializeFromArgument into library

	public static abstract class FetchArgument {
		public abstract Object fetch(String description);
	}

	public static abstract class StringArgConverter {
		public abstract Object convert(String input);
	}

	public static Object initializeFromArgument(
		String[] args, int index,
		FetchArgument fetch, String description,
		StringArgConverter converter
	) {
		Object output = null;
		String obtained = null;
		try {
			obtained = args[index];
		} catch (Exception ex) { }
		if (obtained != null && !obtained.isEmpty()) {
			output = converter.convert(obtained);
		}
		if (output == null)
			output = fetch.fetch(description);
		return output;
	}

	private static File fileHelper(String title, int mode) {
		java.awt.FileDialog chooser = new java.awt.FileDialog(
			new java.awt.Frame(), title, mode
		);
		chooser.setVisible(true);
		chooser.setLocationRelativeTo(null);

		String directory = chooser.getDirectory();
		String file = chooser.getFile();

		if (directory == null || file == null) { return null; }
		return new File(directory + file);
	}

	public static File showOpenFileDialog(String what) {
		return fileHelper(
			"Select " + what + " for opening",
			java.awt.FileDialog.LOAD
		);
	}

	public static File showSaveFileDialog(String what) {
		return fileHelper(
			"Select path to save " + what,
			java.awt.FileDialog.SAVE
		);
	}

	public static byte[] readAllBytes(File file) {
		byte[] fileData;
		FileInputStream fileInputStream = null;
		fileData = new byte[(int)file.length()];
		try {
			fileInputStream = new FileInputStream(file);
			fileInputStream.read(fileData);
		} catch (Exception e) {}
		try { fileInputStream.close(); } catch (Exception e) {}
		return fileData;
	}

	public static byte[] getFileData(String description) {
		File file = showOpenFileDialog(description);
		return readAllBytes(file);
	}

	public static void writeAllBytes(File outputDest, byte[] output) {
		FileOutputStream outputStream = null;
		try {
			outputStream = new FileOutputStream(outputDest);
			outputStream.write(output);
		} catch (Exception e) {}
		try { outputStream.close(); } catch (Exception e) {}
	}

	public static void putFileData(String description, byte[] output) {
		File outputDest = showSaveFileDialog(description);
		writeAllBytes(outputDest, output);
	}

	public static byte[] fileGetter(
		String[] args, int index, String desc
	) {
		return (byte[])initializeFromArgument(
			args, index,
			new FetchArgument() {
				@Override
				public Object fetch(String description) {
					return getFileData(description);
				}
			},
			desc,
			new StringArgConverter() {
				@Override
				public Object convert(String input) {
					return readAllBytes(new File(input));
				}
			}
		);
	}

	public static void fileSaver(
		String[] args, int index, String desc, final byte[] outputData
	) {
		initializeFromArgument(
			args, index,
			new FetchArgument() {
				@Override
				public Object fetch(String description) {
					putFileData(description, outputData);
					return true;
				}
			},
			desc,
			new StringArgConverter() {
				@Override
				public Object convert(String input) {
					writeAllBytes(new File(input), outputData);
					return true;
				}
			}
		);
	}

	// Thanks to Zahlman
	public static byte[] bytesToBytes(byte[]... arrays) {
		int size = 0;
		for (byte[] array: arrays) { size += array.length; }
		byte[] result = new byte[size];

		int position = 0;
		for (byte[] array: arrays) {
			System.arraycopy(array, 0, result, position, array.length);
			position += array.length;
		}

		return result;
	}

	// Use this to make drivers
	private static abstract class Run {
		public abstract void execute(String[] args);
	}

	// Methods for program logic

	// This is for passing integers by reference
	// and should be replaced with a library reference
	public static final class IntWrap {
		int theInt;

		public IntWrap(int input) {
			setValue(input);
		}

		public int value() { return theInt; }

		public IntWrap setValue(int input) {
			theInt = input;
			return this;
		}

		public IntWrap change(int input) {
			theInt += input;
			return this;
		}
	}

	// Tested and working
	public static long readVLI(byte[] input, IntWrap loc) {
		// Read number of leading 1 bits to determine VLI length
		int significantBytes = 1;
		while (input[loc.value()] == (byte)0xFF) {
			significantBytes += 8;
			loc.change(1);
		}
		int bitPos = 0x80;
		while ((input[loc.value()] & bitPos) != 0) {
			significantBytes++;
			bitPos >>= 1;
		}
		long output = 0;
		if (bitPos == 1)
			loc.change(1);
		else {
			// Read uppermost significant bits when they appear
			// somewhere other than the most significant position
			// of their byte
			output |=
				((long)(bitPos - 1) & input[loc.value()])
				<< ((significantBytes - 1) * 8);
			loc.change(1);
			significantBytes--;
		}
		// Read the rest of the significant bits
		while (significantBytes > 0) {
			output |=
				((long)(0xFF) & input[loc.value()])
				<< ((significantBytes - 1) * 8);
			loc.change(1);
			significantBytes--;
		}

		return output;
	}

	// Tested and working
	public static byte[] toVLI(long input) {
		int significantBits = 64 - Long.numberOfLeadingZeros(input);
		// Prevents edge case leading to encoding failure and data loss?
		if (significantBits % 8 == 0)
			significantBits++;
		int significantBytes = (int)Math.ceil(significantBits/8.0);
		while (true) {
			int spareBits = significantBytes * 8 - significantBits;
			// Consider more leading 0 bits to be significant as
			// necessary until an equilibrium between length encoding
			// bits and significant bits is reached
			if (
				spareBits < significantBytes % 8
			) {
				significantBits++;
				if (significantBits % 8 == 1)
					significantBytes++;
			}
			else break;
		}
		final int leadingBytes = (int)Math.ceil(
			(significantBits - 7 * significantBytes)/8.0
		);
		// Create the VLI
		byte[] output = new byte[leadingBytes + significantBytes];
		// Insert the significant bits
		for (int i = 0; i < significantBytes; i++)
			output[output.length - i - 1] = (byte)(input >> (i * 8));
		int offset = 0;
		int oneBits = significantBytes - 1;
		int bitPos = 0x80;
		// Set the upper bits of the VLI to encode the length
		while (oneBits > 8) {
			output[offset++] = (byte)0xFF;
			oneBits -= 8;
		}
		while (oneBits > 0) {
			output[offset] |= bitPos;
			bitPos >>= 1;
			oneBits--;
		}
		return output;
	}

	// Drivers

	// Tested and working
	private static class PatchApplier extends Run {
		public void execute(String[] args) {
			byte[] patch = fileGetter(args, 1, "JFP file");
			byte[] fileToPatch = fileGetter(args, 2, "target file");
			final int md5Length = 16;
			IntWrap offset = new IntWrap(md5Length << 1);
			// Read unaltered MD5
			byte[] originalDigestArray =
				java.util.Arrays.copyOf(patch, md5Length);
			// Read altered MD5
			byte[] alteredDigestArray =
				java.util.Arrays.copyOfRange(patch, md5Length, md5Length << 1);
			// Read length expected of file being patched
			final int fileLength = (int)readVLI(patch, offset);
			System.out.println(String.format("File length: %08X", fileLength));
			System.out.println();
			// Extend the file being patched if necessary to meet the expected size
			if (fileToPatch.length < fileLength) {
				//throw new RuntimeException("File being patched is too short!");
				fileToPatch = java.util.Arrays.copyOf(fileToPatch, fileLength);
			}
			MessageDigest originalDigest = null;
			MessageDigest alteredDigest = null;
			try {
				originalDigest = MessageDigest.getInstance("MD5");
				alteredDigest = MessageDigest.getInstance("MD5");
				originalDigest.reset();
				alteredDigest.reset();
			} catch (Exception e) {
				throw new RuntimeException("MD5 digest classes failed to initialize");
			}
			// For decoding the relative offsets into absolute addresses
			int pos = 0;
			while (offset.value() < patch.length) {
				// Read relative offset and decode it
				int targetOffset = (int)readVLI(patch, offset) + pos;
				System.out.println(String.format("%08X", targetOffset));
				// Read length of patch string
				int length = (int)readVLI(patch, offset);
				System.out.println(String.format("%08X", length));
				System.out.println();
				// Update offset decoding variable
				pos = targetOffset + length;
				// Read patch data
				byte[] patchData = java.util.Arrays.copyOfRange(
					patch, offset.value(), offset.value() + length
				);
				// Patch file and update digests
				for (int i = 0; i < length; i++) {
					originalDigest.update(fileToPatch[targetOffset + i]);
					fileToPatch[targetOffset + i] ^= patchData[i];
					alteredDigest.update(fileToPatch[targetOffset + i]);
				}
				// Advance file cursor
				offset.change(length);
			}
			// Determine integrity of patch application
			byte[] newOriginalDigestArray = originalDigest.digest();
			byte[] newAlteredDigestArray = alteredDigest.digest();
			boolean patchSafe = java.util.Arrays.equals(
				originalDigestArray, newOriginalDigestArray
			) && java.util.Arrays.equals(
				alteredDigestArray, newAlteredDigestArray
			);
			boolean reversionSafe = java.util.Arrays.equals(
				alteredDigestArray, newOriginalDigestArray
			) && java.util.Arrays.equals(
				originalDigestArray, newAlteredDigestArray
			);
			if (!patchSafe && !reversionSafe) {
				/**
				 * Print MD5 information
				System.out.println("Busted MD5s.");
				System.out.println("Had { ");
				for (int i = 0; i < 16; i++)
					System.out.print(String.format("%02X ", newOriginalDigestArray[i]));
				System.out.println("}");
				System.out.print("Expected { ");
				for (int i = 0; i < 16; i++)
					System.out.print(String.format("%02X ", originalDigestArray[i]));
				System.out.println("}");
				System.out.println("Had { ");
				for (int i = 0; i < 16; i++)
					System.out.print(String.format("%02X ", newAlteredDigestArray[i]));
				System.out.println("}");
				System.out.print("Expected { ");
				for (int i = 0; i < 16; i++)
					System.out.print(String.format("%02X ", alteredDigestArray[i]));
				System.out.println("}");
				**/
				throw new RuntimeException(
					"The provided patch is not safe for the given file!"
				);
			}
			final byte[] outputData = java.util.Arrays.copyOf(
				fileToPatch, fileToPatch.length
			);
			fileSaver(args, 3, "patched file", outputData);
		}
	}

	// Tested and working
	private static class PatchMaker extends Run {
		public void execute(String[] args) {
			byte[] original = fileGetter(args, 1, "original file");
			byte[] altered = fileGetter(args, 2, "altered file");
			// Determine which file is bigger and use the largest
			// file's size as the size of the file resuting from
			// applying this patch
			final int patchSize =
				original.length < altered.length
				? altered.length : original.length;
			// Extend the shorter file to be the size of the longer
			// file if necessary
			if (original.length < patchSize)
				original = java.util.Arrays.copyOf(original, patchSize);
			if (altered.length < patchSize)
				altered = java.util.Arrays.copyOf(altered, patchSize);
			// Encode expected result file size
			byte[] patchSizeVLI = toVLI(patchSize);
			MessageDigest originalDigest = null;
			MessageDigest alteredDigest = null;
			try {
				originalDigest = MessageDigest.getInstance("MD5");
				alteredDigest = MessageDigest.getInstance("MD5");
				originalDigest.reset();
				alteredDigest.reset();
			} catch (Exception e) {
				throw new RuntimeException("MD5 digest classes failed to initialize");
			}
			// Difference offset
			int offset = 0;
			// Length of difference
			int length = 0;
			// Accumulated difference data
			ArrayList<Byte> patchData = new ArrayList<Byte>();
			// For storing offsets as relative values
			int lastOffset = 0;
Outer:			while (offset < patchSize) {
				length = 0;
				if (offset >= patchSize)
					break;
				// Find a difference
				while (original[offset] == altered[offset]) {
					offset++;
					if (offset >= patchSize)
						break Outer;
				}
				System.out.println(String.format("%08X", offset));
				// Measure the difference's length
				while (original[offset + length] != altered[offset + length]) {
					length++;
					if (offset + length >= patchSize)
						break;
				}
				System.out.println(String.format("%08X", length));
				System.out.println();
				// Encode the address as a VLI of the offset relative
				// to the address of the byte after the last change
				// (or 0 if not applicable)
				byte[] offsetVLI = toVLI(offset - lastOffset);
				// Encode length of difference
				byte[] lengthVLI = toVLI(length);
				// Add VLIs to patch data
				for (int i = 0; i < offsetVLI.length; i++)
					patchData.add(offsetVLI[i]);
				for (int i = 0; i < lengthVLI.length; i++)
					patchData.add(lengthVLI[i]);
				// Digest the original and altered values of the
				// bytes to be changed and add the exclusive
				// or of the original and altered values to the
				// patch data
				for (int i = 0; i < length; i++) {
					byte originalByte = original[offset + i];
					byte alteredByte = altered[offset + i];
					originalDigest.update(originalByte);
					alteredDigest.update(alteredByte);
					patchData.add((byte)(originalByte ^ alteredByte));
				}
				// Update offset encoding meta data
				lastOffset = offset += length;
			}
			// Write digests and expected length of result file
			byte[] tempArray = bytesToBytes(
				originalDigest.digest(),
				alteredDigest.digest(),
				patchSizeVLI
			);
			// Write patch data
			final byte[] output = new byte[tempArray.length + patchData.size()];
			System.arraycopy(tempArray, 0, output, 0, tempArray.length);
			int i = 0;
			for (byte curr: patchData)
				output[tempArray.length + (i++)] = curr;
			fileSaver(args, 3, "patch", output);
		}
	}

	// Tested and working
	private static class VLITester extends Run {
		public void execute(String[] args) {
			long val = 0x7FFFFFFFFFFFFFFFL;
			do {
				byte[] asVLI = toVLI(val);
				long test = readVLI(asVLI, new IntWrap(0));
				System.out.println("Value is:\t" + val);
				System.out.println(
					"Became:\t\t" + test
				);
				System.out.print("{ ");
				for (int i = 0; i < asVLI.length; i++)
					System.out.print(
						String.format("%02X ", asVLI[i])
					);
				System.out.println("}");
				System.out.println(
					"Are they the same? "
					+ (val == test)
				);
				System.out.println(
					"VLI length in bytes: "
					+ asVLI.length
				);
				System.out.println();
				if (val == 0) break;
				val >>= 1;
				if (val < 0)
					val &= 0x7FFFFFFFFFFFFFFFL;
			} while (true);
		}
	}

	// Change this and build again to use a different tool,
	// or change it programmatically before calling run.execute(args)
	// to simulate multiple utilities
	private static Run run = new PatchApplier();

	// Main

	public enum Mode { MAKE, APPLY, TEST_VLI };

	private static Mode askMode() {
		final int option = JOptionPane.showConfirmDialog(
			null,
			"Are you creating a patch instead of applying one?",
			"JFP by Hextator",
			JOptionPane.YES_NO_OPTION
		);
		if (option == JOptionPane.YES_OPTION)
			return Mode.MAKE;
		return Mode.APPLY;
	}

	public static void main(String[] args) {
		try {
			UIManager.setLookAndFeel(
				UIManager.getSystemLookAndFeelClassName()
			);
			Mode selected = null;
			selected = (Mode)initializeFromArgument(
				args, 0,
				new FetchArgument() {
					@Override
					public Object fetch(String description) {
						return askMode();
					}
				},
				"patching mode",
				new StringArgConverter() {
					@Override
					public Object convert(String input) {
						return Mode.valueOf(input);
					}
				}
			);
			switch (selected) {
				case APPLY:
					break;
				case TEST_VLI:
					run = new VLITester();
					break;
				case MAKE:
				default:
					run = new PatchMaker();
					break;
			}
			run.execute(args);
		} catch (Exception e) {
			final Writer result = new StringWriter();
			final PrintWriter printWriter = new PrintWriter(result);
			e.printStackTrace(printWriter);
			JTextArea errorDisplayArea = new JTextArea(
				"Exception: " +
				e.getMessage() +
				"\n" +
				result.toString()
			);
			JScrollPane errorPane = new JScrollPane(errorDisplayArea);
			JOptionPane.showMessageDialog(
				null,
				errorPane,
				"Error",
				JOptionPane.ERROR_MESSAGE
			);
		}
		System.exit(0);
	}
}
